#define SunStrength 3
#define BumpSmoothness 4
#define MaxReflection 0.6

struct VS_INPUT 
{
	float4 	Position   : POSITION;
	int4   	BatchIndex : BLENDINDICES;
    half2	TexCoord   : TEXCOORD0;
    float3	Normal     : NORMAL;
};

struct VS_OUTPUT
{
    float4 	Position 	: POSITION;
    float2 	TexCoord 	: TEXCOORD0;
    float4 	WorldPos 	: TEXCOORD1;
	float3 	Normal		: TEXCOORD2;
	float3 	InvViewDir	: TEXCOORD3;
	float	FogAmount	: TEXCOORD4;
	float3	PointLVis[2] : TEXCOORD5;
	row_major float2x4 Material : COLOR;
};

#define ViewSpaceLight
#include "Light.fx"
#include "Fog.fx"

typedef float3x3 LightColor;
#define LAmbient 0
#define LDiffuse 1
#define LSpecular 2

bool bSpecularEnable : SPECULARENABLE;
float4 TFactor : TFACTOR;
bool bUseTFactorColor;
bool bUseTFactorAlpha;

bool bUseTexture;
bool bReflective;
bool bSpecularMap;
bool bBumpMap;

static const float AlphaThreshold = 8.0/255.0;

float4x4 World : WORLD;
float4x4 View : VIEW;
float4x4 ViewProj : VIEWPROJ;

#ifndef MAX_BATCH_SIZE
	#define MAX_BATCH_SIZE 16
#endif

row_major float4x4 arrWorldTx[MAX_BATCH_SIZE];
row_major float2x4 arrMaterial[MAX_BATCH_SIZE];

struct CMaterial
{
	float4 vDiffuse;
	float3 vSpecular;
	float fPower;
};

static CMaterial Material;

float4x4 ViewProjLight[4];
bool bStaticSplit;
int iStaticSplit;

#ifndef SMAP_SIZE
	#define SMAP_SIZE 2048
#endif
#ifndef SHADOW_FILTER
	#define SHADOW_FILTER 1
#endif

sampler2D Tex0 : register( s0 ); // main texture
sampler2D Tex1 : register( s1 ); // gloss map 
samplerCUBE TexEnv : register( s4 ); // environment
sampler2D TexBump : register( s5 ); // main texture for bump mapping (different sampler settings)
sampler2D TexShadow : register( s6 ); // shadowmap

VS_OUTPUT vs_main( in VS_INPUT In )
{
    VS_OUTPUT Out;
	
	float4x4 mWorld = arrWorldTx[In.BatchIndex.x];
	
    float4 worldPos = mul(In.Position, mWorld);
    Out.Position = mul(worldPos, ViewProj); 
    Out.TexCoord = In.TexCoord;
    Out.WorldPos = worldPos;

	// Normal in world space
	Out.Normal = (mul(In.Normal, (float3x3)mWorld));

	// Position in view space
    float3 viewPos = mul(worldPos, View).xyz;
	
	Out.InvViewDir = -viewPos;
	Out.FogAmount = CalcFog(viewPos.z);

	Out.Material = arrMaterial[In.BatchIndex.x];

	// Point lights
	for(int i = 0; i < 6; i++)
	{
		float vis = 0;
		if(i < iNLights)
		{
			CPointLight light = arrLights[i];
			float dist = length(light.vWorldPos - worldPos.xyz); 
			// check maximum distance
			vis = saturate(light.vAttenuation.w - dist);
		}
		Out.PointLVis[i<3?0:1][i<3?i:i-3] = vis;
	}
	
    return Out;
}

//////////////////////

// Calculates a cotangent frame without precomputed tangents by Christian Schuler
// ported from GLSL to HLSL; see: http://www.thetenthplanet.de/archives/1180
float3x3 calcWsCotangentFrame (float3 wsNormal, float3 wsInvViewDir, float2 tsCoord)
{
    // get edge vectors of the pixel triangle
    float3 dp1 = ddx(wsInvViewDir);
    float3 dp2 = ddy(wsInvViewDir);
    float2 duv1 = ddx(tsCoord);
    float2 duv2 = ddy(tsCoord);
 
    // solve the linear system
    float3 dp2perp = cross(dp2, wsNormal);
    float3 dp1perp = cross(wsNormal, dp1);
    float3 T = dp2perp * duv1.x + dp1perp * duv2.x;
    float3 B = dp2perp * duv1.y + dp1perp * duv2.y;
 
    // construct and return a scale-invariant cotangent frame
    float invmax = rsqrt(max(dot(T,T), dot(B,B)));
    return float3x3(T * invmax, B * invmax, wsNormal);
}

static const float2 SHADOW_SIZE = float2(SMAP_SIZE,SMAP_SIZE*4);

float FilterShadowPS(float4 vPosLight, float2 dSdX, float2 dSdY)
{  
    const float2 sample_array[20] = {
        float2(  1.f,  0.f ) / SHADOW_SIZE,
        float2(  0.f,  1.f ) / SHADOW_SIZE,
        float2(  0.f, -1.f ) / SHADOW_SIZE,
        float2( -1.f,  0.f ) / SHADOW_SIZE,

        float2( -1.f, -1.f ) / SHADOW_SIZE,
        float2(  1.f, -1.f ) / SHADOW_SIZE,
        float2( -1.f,  1.f ) / SHADOW_SIZE,
        float2(  1.f,  1.f ) / SHADOW_SIZE,       

        float2(  0.f, -2.f ) / SHADOW_SIZE,
        float2( -2.f,  0.f ) / SHADOW_SIZE,
        float2(  0.f,  2.f ) / SHADOW_SIZE,
        float2(  2.f,  0.f ) / SHADOW_SIZE,

        float2( -1.f, -2.f ) / SHADOW_SIZE,
        float2(  1.f, -2.f ) / SHADOW_SIZE,
        float2( -2.f, -1.f ) / SHADOW_SIZE,
        float2( -2.f,  1.f ) / SHADOW_SIZE,
        
        float2( -1.f,  2.f ) / SHADOW_SIZE,
        float2(  1.f,  2.f ) / SHADOW_SIZE,
        float2(  2.f, -1.f ) / SHADOW_SIZE,
        float2(  2.f,  1.f ) / SHADOW_SIZE,
     };

#if SHADOW_FILTER == 1	 
    float approx_major_len = max( dot(dSdX, dSdX), dot(dSdY, dSdY) );
    float rho = 0.5f*log2(approx_major_len);
#endif

    float shadow = tex2Dlod(TexShadow, vPosLight).x;
    float4 offset = 0;
       
    for ( int i=0; i<4; i++ )
    {
        offset.x = dot(sample_array[i].xy, float2(0.793353,-0.608761)*1.2);
        offset.y = dot(sample_array[i].xy, float2(0.608761, 0.793353)*1.2);
        shadow += tex2Dlod(TexShadow, vPosLight + offset).x;
    }
#if SHADOW_FILTER == 1	
    if ( rho < 0.f )
    {
        float shadow2 = shadow;
        
        for ( int i=4; i<20; i++ )
        {
            offset.x = dot(sample_array[i].xy, float2(0.793353,-0.608761)*1.2);
            offset.y = dot(sample_array[i].xy, float2(0.608761, 0.793353)*1.2);
            shadow2 += tex2Dlod(TexShadow, vPosLight + offset).x;
        }
        shadow2 = shadow + shadow2;
        shadow2 = smoothstep(1, 19, shadow2);
        shadow = smoothstep(1, 5, shadow);
     
        float lerp_fac = saturate(-rho);
                     
        shadow = lerp(shadow, shadow2, lerp_fac);
    }
    else
#endif
    {
        shadow = smoothstep(1,5,shadow);
    }
	
	return shadow;
}

float4 calcShadowPos(float4 WorldPos, float2 ShadowTexC, float4 vPosLight, float slice)
{
	ShadowTexC.y = 1.0 - ShadowTexC.y;
	ShadowTexC.y += slice;
	ShadowTexC.y /= 4.0;
	return float4(ShadowTexC.xy, vPosLight.z/vPosLight.w, 0.0);
}

float calcLightAmountDynamic(float4 WorldPos, float NdotL)
{
	const float Half = 0.5;
	float4 vPosLight;
	float2 ShadowTexC;
	int i;
	for(i = 0; i < 4; i++)
	{
		vPosLight = mul(WorldPos, ViewProjLight[i]);
		ShadowTexC = Half * vPosLight.xy / vPosLight.w + Half.xx;
		if(ShadowTexC.y - 1.0f/SMAP_SIZE >= 0 && ShadowTexC.x >= 0 &&
			ShadowTexC.y + 1.0f/SMAP_SIZE < 1 && ShadowTexC.x <= 1)
		{
			break;
		}		
	}
	
	if(i == 4) 
	{
		return 1;
	}
	else
	{
		float4 samplePos = calcShadowPos(WorldPos, ShadowTexC, vPosLight, i);
		float2 dSdX = SHADOW_SIZE * ddx( samplePos.xy );
		float2 dSdY = SHADOW_SIZE * ddy( samplePos.xy );
		if(NdotL > 0)
		{
			if(i < 3)
				return FilterShadowPS(samplePos, dSdX, dSdY);
			else
				return tex2Dlod( TexShadow, samplePos ).x;
		}
		else
		{
			return 0;
		}
	}
}

float calcLightAmountStatic(float4 WorldPos, float NdotL)
{
	const float Half = 0.5;
	float4 vPosLight = mul(WorldPos, ViewProjLight[iStaticSplit]);
	float2 ShadowTexC = Half * vPosLight.xy / vPosLight.w + Half.xx;
	float4 samplePos = calcShadowPos(WorldPos, ShadowTexC, vPosLight, iStaticSplit);
	float2 dSdX = SHADOW_SIZE * ddx( samplePos.xy );
	float2 dSdY = SHADOW_SIZE * ddy( samplePos.xy );
	if(NdotL > 0)
	{
		if(iStaticSplit < 3)
			return FilterShadowPS(samplePos, dSdX, dSdY);
		else
			return tex2Dlod( TexShadow, samplePos ).x;
	}
	else
	{
		return 0;
	}
}

void CalcPointLight(inout LightColor color, CPointLight light, float4 worldPos, float3 viewNormal, float3 invViewDir) 
{ 
	float3 wL = light.vWorldPos - worldPos.xyz;
	float dist = length(wL); 
	if(dist > light.vAttenuation.w)
		return;

	float3 viewDirLight = normalize(mul(wL, (float3x3)View));
	float NdotL = dot(viewNormal, viewDirLight);
	if(NdotL > 0)
	{ 
		// Attenuation
		float lightAmount = saturate(NdotL * 30) / (light.vAttenuation.x + (light.vAttenuation.y + light.vAttenuation.z * dist) * dist);

		// Diffuse
		color[LDiffuse] += NdotL * lightAmount * light.vDiffuse; 
	
		// Specular
		if(bSpecularEnable) 
		{ 
			float3 H = normalize(viewDirLight + invViewDir);
			color[LSpecular] += pow(saturate(dot(H, viewNormal)), Material.fPower) * lightAmount * light.vSpecular;
		}
	}
}

void CalcDirLight(inout LightColor color, CDirLight light, float3 viewNormal, float3 invViewDir, float glossMap) 
{ 
	float NdotL = dot(viewNormal, light.vViewDir);
	if(NdotL > 0) 
	{ 
		// Avoid excessively sharp light cut offs
		float lightAmount = saturate(NdotL * 30);
		
		//Diffuse
		color[LDiffuse] += NdotL * lightAmount * light.vDiffuse; 

		//Specular
		if(bSpecularEnable) 
		{ 
			float3 H = normalize(light.vViewDir + invViewDir);
			float HdotN = saturate(dot(H, viewNormal));
			color[LSpecular] += pow(HdotN, Material.fPower) * glossMap * lightAmount * light.vSpecular;
		}
	}
}

void CalcSunLight(inout LightColor color, CDirLight light, float4 worldPos, float3 viewNormal, float3 invViewDir, float4 mainTex, float glossMap, bool bShadow) 
{ 
	float NdotL = dot(viewNormal, light.vViewDir);
	if(NdotL > 0) 
	{ 
		float lightAmount = 1;

		if(bShadow)
		{
			//[branch] if(bStaticSplit)
			//	lightAmount = calcLightAmountStatic(worldPos, NdotL);
			//else
				lightAmount = calcLightAmountDynamic(worldPos, NdotL);
		}

		// Avoid excessively sharp light cut offs
		lightAmount *= saturate(NdotL * 30);
		// Reduce shadows on semi-transparent surfaces
		float lightAmountFull = lightAmount;
		lightAmount = saturate(lightAmount + (1 - mainTex.a));
		if(lightAmount > 0)
		{
			//Diffuse
			color[LDiffuse] += NdotL * lightAmount * light.vDiffuse; 

			//Specular
			if(bSpecularEnable || bReflective) 
			{ 
				// Specular still uses full shadow disregarding alpha adjustments (because specular will increase alpha)
				if(lightAmountFull > 0)
				{
					float3 H = normalize(light.vViewDir + invViewDir);
					float HdotN = saturate(dot(H, viewNormal));
					color[LSpecular] += 
						(
							pow(HdotN, Material.fPower) + 
							(Material.fPower > 5 ? pow(HdotN, 600 / (1 + glossMap)) * SunStrength : 0)
						) * glossMap * lightAmountFull * 
						(bReflective ? lerp(1, mainTex.rgb, glossMap) * light.vSpecular: light.vSpecular);
				}
			}
		}
	}
}

float4 ps_main ( 
	in VS_OUTPUT In, 
	uniform bool bBump,
	uniform bool bShadow,
	uniform bool bEnvMap,
	uniform bool bPointLight
	) : COLOR
{
	Material.vDiffuse = In.Material[0].rgba;
	Material.vSpecular = In.Material[1].rgb;
	Material.fPower = In.Material[1].a;

	// Renormalize interpolated normals
	In.Normal = normalize(In.Normal);
	In.InvViewDir = normalize(In.InvViewDir);

	// Load texture
	float4 mainTex = bUseTexture ? tex2D(Tex0, In.TexCoord) : 1;
	if(bUseTFactorColor) mainTex.rgb = TFactor.rgb;
	if(bUseTFactorAlpha) mainTex.a = TFactor.a;

	float glossMap = bSpecularMap || bReflective ? tex2D(Tex1, In.TexCoord).g : 1;
	if(bReflective) glossMap = saturate(glossMap + 0.2);

	// Apply material's alpha
	mainTex.a *= Material.vDiffuse.a;
	//mainTex.rgb = 1;
	//return mainTex;
		
	if(bBump)
	{
		if(bBumpMap)
		{
			const float2 dUV = float2(1.0f/2048.0f,0);
		
			float4x3 bumpColor;
			bumpColor[0] = tex2D(TexBump, In.TexCoord-dUV.xy).rgb;
			bumpColor[1] = tex2D(TexBump, In.TexCoord-dUV.yx).rgb;
			bumpColor[2] = tex2D(TexBump, In.TexCoord+dUV.xy).rgb;
			bumpColor[3] = tex2D(TexBump, In.TexCoord+dUV.yx).rgb;
			// Convert to "elevation"
			float4 bumpValue;
			bumpValue.x = max(max(bumpColor[0].r,bumpColor[0].g),bumpColor[0].b);
			bumpValue.y = max(max(bumpColor[1].r,bumpColor[1].g),bumpColor[1].b);
			bumpValue.z = max(max(bumpColor[2].r,bumpColor[2].g),bumpColor[2].b);
			bumpValue.w = max(max(bumpColor[3].r,bumpColor[3].g),bumpColor[3].b);
			bumpValue *= mainTex.a;
			bumpValue = bumpValue*(2 - bumpValue);
		
			float3 tsBumpNormal = normalize(float3(bumpValue.zw-bumpValue.xy, BumpSmoothness));
			float3x3 wtx = calcWsCotangentFrame(In.Normal, -In.WorldPos.xyz, In.TexCoord);
			In.Normal = normalize(mul(tsBumpNormal, wtx));
		}
	}

	// Normal in view space
	float3 viewNormal = normalize(mul(In.Normal, (float3x3)View));

	// Initialize lights
	LightColor light;
	light[LAmbient] = vAmbientLight;
	light[LDiffuse] = 0;
	light[LSpecular] = 0;
/*
	float3 reflDir = reflect(In.WorldPos.xyz, In.Normal.xyz);
	float3 envColor = texCUBElod(TexEnv, float4(reflDir,8)).rgb;
	light[LAmbient] = vAmbientLight * (0.7 + envColor*0.6);
*/
//	return float4(envColor, mainTex.a);
	
	// Sun/Moon always present
	CalcSunLight(light, Sun, In.WorldPos, viewNormal, In.InvViewDir, mainTex, glossMap, bShadow);

	// Cockpit Light
	if(bCockpitLight)
		CalcDirLight(light, Cockpit, viewNormal, In.InvViewDir, glossMap);
		
	// Point Lights
	if(bPointLight)
	{
		[branch] if(iNLights > 0)
		{
			[unroll] for(int iP = 0; iP < iNLights; iP++)
				[branch] if(In.PointLVis[iP/3][iP%3] > 0)
					CalcPointLight(light, arrLights[iP], In.WorldPos, viewNormal, In.InvViewDir);
		}
	}

	// Apply material sensitivity to specific light
	light[LAmbient] *= Material.vDiffuse.rgb;
	light[LDiffuse] *= Material.vDiffuse.rgb;
	light[LSpecular] *= Material.vSpecular * glossMap;

	float4 Color;
	
	// Must use saturate to match original lighting
	Color.rgb = saturate(light[LAmbient] + light[LDiffuse]) * mainTex.rgb;
	Color.a = mainTex.a;
	
	if(Color.a > AlphaThreshold)
		Color.a += light[LSpecular].g * saturate(mainTex.a*20 - 160.0/255);

	if(bEnvMap)
	{
		float fr0 = bReflective ? glossMap : dot(Material.vSpecular, 0.01);
		// Using non-interpolated normal, this is generally redundant calculations and worse quality,  
		//  but some normals in existing models are really screwed by interpolation (propellers, or Spad13 upper wing padding).
		float3 wN = normalize(cross(ddx(In.WorldPos.xyz), ddy(In.WorldPos.xyz)));
		//float3 wN = In.Normal.xyz;
		float fr = fr0 + pow(1-saturate(dot(wN, -normalize(In.WorldPos.xyz))), 5) * (1 - fr0);
		float3 reflDir = reflect(In.WorldPos.xyz, In.Normal.xyz);
		float4 cubeVec = float4(reflDir, 8 * saturate(1-log2(Material.fPower)/5));
		float3 envColor = vAmbientLight * texCUBElod(TexEnv, cubeVec).rgb;
		if(bReflective)
		{
			Color.rgb = Color.rgb * (1 - fr*glossMap) + envColor * mainTex.rgb * fr * glossMap;
		}
		else
		{
			fr = min(fr, dot(Material.vSpecular, MaxReflection*0.33)) * glossMap;
			Color.rgb = Color.rgb * (1 - fr) + envColor * fr;
		}
	}
	
	// Add specular highlight
	Color.rgb += light[LSpecular];
	
	ApplyFog(Color, In.FogAmount);
	
	return Color;
}

float4 ps_reflSight	(
	in VS_OUTPUT In 
	) : COLOR
{
	float4 mainTex = tex2D(Tex0, In.TexCoord);
	mainTex.a *= 0.5 * In.Material[0].a;
	return mainTex;
}

/////////////////////////////////////////

technique ReflSight
{
	pass Pass1
	{
		VertexShader = compile vs_3_0 vs_main();
		PixelShader = compile ps_3_0 ps_reflSight();
	}
}

technique PropDisk
{
	pass Pass1
	{
		VertexShader = compile vs_3_0 vs_main();
		PixelShader = compile ps_3_0 ps_main( false, false, false, false );
	}
}

technique Standard
{
	pass Pass1
	{
		VertexShader = compile vs_3_0 vs_main();
		PixelShader = compile ps_3_0 ps_main( true, SMAP_SIZE > 0, true, true );
	}
}